<?php

namespace App\Http\Controllers;

use App\Models\Color;
use Illuminate\Http\Request;

class ColorController extends Controller
{
    /**
     * Display a listing of colors.
     */
    public function index(Request $request)
    {
        try {
            $query = Color::query();

            // 🔍 Search filter
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where('name', 'like', "%{$search}%");
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $colors = $query->paginate($perPage);

            // Transform color data
            $colors->getCollection()->transform(function ($color) {
                return [
                    'id' => $color->id,
                    'name' => $color->name,
                    'status' => $color->status,
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Colors retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $colors->total(),
                    'total_page' => $colors->lastPage(),
                    'current_page' => $colors->currentPage(),
                    'current_page_count' => $colors->count(),
                    'next_page' => $colors->hasMorePages() ? $colors->currentPage() + 1 : null,
                    'previous_page' => $colors->onFirstPage() ? null : $colors->currentPage() - 1,
                ],
                'data' => $colors->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }


    /**
     * Store a newly created color.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'status' => 'nullable|in:active,inactive',
            ]);

            $color = Color::create([
                'name' => $request->name,
                'status' => $request->status ?? 'active',
            ]);

            return response()->json([
                'status'=>true,
                'message' => 'Color created successfully',
                'data' => $color
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified color.
     */
    public function show($id)
    {
        try {
            $color = Color::find($id);

            if (!$color) {
                return response()->json(['message' => 'Color not found'], 404);
            }

            return response()->json(['status'=>true,'data' => $color]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified color.
     */
    public function update(Request $request, $id)
    {
        try {
            $color = Color::find($id);

            if (!$color) {
                return response()->json(['message' => 'Color not found'], 404);
            }

            $request->validate([
                'name' => 'nullable|string|max:255',
                'status' => 'nullable|in:active,inactive',
            ]);

            $color->name = $request->name ?? $color->name;
            $color->status = $request->status ?? $color->status;
            $color->save();

            return response()->json([
                'status'=>true,
                'message' => 'Color updated successfully',
                'data' => $color
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified color.
     */
    public function destroy($id)
    {
        try {
            $color = Color::find($id);

            if (!$color) {
                return response()->json(['message' => 'Color not found'], 404);
            }

            $color->delete();

            return response()->json(['status'=>true,'message' => 'Color deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
