<?php

namespace App\Http\Controllers;

use App\Models\Contact;
use Illuminate\Http\Request;

class ContactController extends Controller
{
    /**
     * Display a listing of contacts.
     */
    public function index(Request $request)
    {
        try {
            $query = Contact::query();

            // 🔍 Search filter by name, email, or subject
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('subject', 'like', "%{$search}%");
                });
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $contacts = $query->paginate($perPage);

            // Transform contact data
            $contacts->getCollection()->transform(function ($contact) {
                return [
                    'id' => $contact->id,
                    'name' => $contact->name,
                    'email' => $contact->email,
                    'subject' => $contact->subject,
                    'message' => $contact->message,
                    'created_at' => $contact->created_at->toDateTimeString(),
                    'updated_at' => $contact->updated_at->toDateTimeString(),
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Contacts retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $contacts->total(),
                    'total_page' => $contacts->lastPage(),
                    'current_page' => $contacts->currentPage(),
                    'current_page_count' => $contacts->count(),
                    'next_page' => $contacts->hasMorePages() ? $contacts->currentPage() + 1 : null,
                    'previous_page' => $contacts->onFirstPage() ? null : $contacts->currentPage() - 1,
                ],
                'data' => $contacts->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created contact.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'email' => 'required|email|max:255',
                'subject' => 'required|string|max:255',
                'message' => 'required|string',
            ]);

            $contact = Contact::create($request->only('name', 'email', 'subject', 'message'));

            return response()->json([
                'message' => 'Contact submitted successfully',
                'data' => $contact
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified contact.
     */
    public function show($id)
    {
        try {
            $contact = Contact::find($id);

            if (!$contact) {
                return response()->json(['message' => 'Contact not found'], 404);
            }

            return response()->json(['data' => $contact]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified contact.
     */
    public function update(Request $request, $id)
    {
        try {
            $contact = Contact::find($id);

            if (!$contact) {
                return response()->json(['message' => 'Contact not found'], 404);
            }

            $request->validate([
                'name' => 'nullable|string|max:255',
                'email' => 'nullable|email|max:255',
                'subject' => 'nullable|string|max:255',
                'message' => 'nullable|string',
            ]);

            $contact->update($request->only('name', 'email', 'subject', 'message'));

            return response()->json([
                'message' => 'Contact updated successfully',
                'data' => $contact
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified contact.
     */
    public function destroy($id)
    {
        try {
            $contact = Contact::find($id);

            if (!$contact) {
                return response()->json(['message' => 'Contact not found'], 404);
            }

            $contact->delete();

            return response()->json(['message' => 'Contact deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
