<?php

namespace App\Http\Controllers;

use App\Models\Product;
use App\Models\Purchase;
use Illuminate\Http\Request;

class PurchaseController extends Controller
{
    /**
     * Display a listing of purchases.
     */
    public function index(Request $request)
    {
        try {
            $query = Purchase::with('supplier');

            // Search by purchase_id or supplier name/phone
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('purchase_id', 'like', "%{$search}%")
                        ->orWhereHas('supplier', function ($supplierQuery) use ($search) {
                            $supplierQuery->where('name', 'like', "%{$search}%")
                                ->orWhere('phone', 'like', "%{$search}%");
                        });
                });
            }

            // Add 'days' input to filter from today back N days
            if ($request->filled('days')) {
                $days = (int) $request->get('days');
                if ($days > 0) {
                    $fromDate = now()->subDays($days)->startOfDay();
                    $toDate = now()->endOfDay();
                    $query->whereBetween('purchase_date', [$fromDate, $toDate]);
                }
            }

            // Filter by date range (from_date / to_date)
            if ($request->filled('from_date')) {
                $from = $request->get('from_date');
                $query->whereDate('purchase_date', '>=', $from);
            }
            if ($request->filled('to_date')) {
                $to = $request->get('to_date');
                $query->whereDate('purchase_date', '<=', $to);
            }

            $perPage = $request->get('per_page', 10);
            $purchases = $query->orderBy('id', 'desc')->paginate($perPage);

            $items = $purchases->getCollection()->transform(function ($purchase) {
                $purchaseArray = $purchase->toArray();

                // Just decode products JSON array, don't add product detail
                $purchaseArray['products'] = json_decode($purchaseArray['products'], true);

                // Return supplier name with supplier_id, instead of entire supplier object
                if ($purchase->supplier) {
                    $purchaseArray['supplier_id'] = $purchase->supplier->id;
                    $purchaseArray['supplier_name'] = $purchase->supplier->name;
                } else {
                    $purchaseArray['supplier_id'] = null;
                    $purchaseArray['supplier_name'] = null;
                }
                // Optionally remove the relation key if you don't want to expose it
                unset($purchaseArray['supplier']);

                return $purchaseArray;
            });

            return response()->json([
                'status' => true,
                'message' => 'Purchases retrieved successfully',
                'data' => $items,
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $purchases->total(),
                    'total_page' => $purchases->lastPage(),
                    'current_page' => $purchases->currentPage(),
                    'current_page_count' => $purchases->count(),
                    'next_page' => $purchases->hasMorePages() ? $purchases->currentPage() + 1 : null,
                    'previous_page' => $purchases->onFirstPage() ? null : $purchases->currentPage() - 1,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created purchase in storage.
     */
    public function store(Request $request)
    {
        try {
            // supplier_id is required as per migration
            $validated = $request->validate([
                'supplier_id'     => 'required|numeric',
                'products'        => 'required|array',
                'total_amount'    => 'required|numeric',
                'vat'             => 'required|numeric',
                'discount'        => 'required|numeric',
                'grand_total'     => 'required|numeric',
                'paid'            => 'nullable|numeric',
                'due'             => 'nullable|numeric',
                'paid_by'         => 'required|string',
                'purchase_date'   => 'required|date',
                'notes'           => 'nullable|string',
            ]);

            // Generate unique purchase_id
            do {
                $purchaseId = 'PUR' . strtoupper(substr(bin2hex(random_bytes(6)), 0, 12));
            } while (Purchase::where('purchase_id', $purchaseId)->exists());

            $validated['purchase_id'] = $purchaseId;
            $validated['products'] = json_encode($validated['products']);

            // set paid and due numeric with default 0 for decimal fields
            $validated['paid'] = array_key_exists('paid', $validated) ? $validated['paid'] : 0;
            $validated['due'] = array_key_exists('due', $validated)
                ? $validated['due']
                : round($validated['grand_total'] - $validated['paid'], 2);

            $purchase = Purchase::create($validated);

            $purchaseArray = $purchase->toArray();
            $purchaseArray['products'] = json_decode($purchaseArray['products'], true);

            return response()->json([
                'status' => true,
                'message' => 'Purchase created successfully',
                'data' => $purchaseArray
            ], 201);
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $ve->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified purchase.
     */
    public function show($id)
    {
        try {
            $purchase = Purchase::find($id);

            if (!$purchase) {
                return response()->json([
                    'status' => false,
                    'message' => 'Purchase not found'
                ], 404);
            }

            $purchaseArray = $purchase->toArray();
            $purchaseArray['products'] = json_decode($purchaseArray['products'], true);

            return response()->json([
                'status' => true,
                'message' => 'Purchase retrieved successfully',
                'data' => $purchaseArray
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified purchase in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $purchase = Purchase::find($id);
            if (!$purchase) {
                return response()->json([
                    'status' => false,
                    'message' => 'Purchase not found'
                ], 404);
            }

            $validated = $request->validate([
                'purchase_id'      => 'sometimes|required|string|unique:purchases,purchase_id,' . $purchase->id,
                'supplier_id'      => 'sometimes|required|string',
                'products'         => 'sometimes|required|array',
                'total_amount'     => 'sometimes|required|numeric',
                'vat'              => 'sometimes|required|numeric',
                'discount'         => 'sometimes|required|numeric',
                'grand_total'      => 'sometimes|required|numeric',
                'paid'             => 'sometimes|numeric',
                'due'              => 'sometimes|numeric',
                'paid_by'          => 'sometimes|required|string',
                'purchase_date'    => 'sometimes|required|date',
                'notes'            => 'nullable|string',
            ]);

            if (array_key_exists('products', $validated) && is_array($validated['products'])) {
                $validated['products'] = json_encode($validated['products']);
            }

            // Numeric and due recalculation logic
            if (isset($validated['grand_total'])) {
                $grand_total = $validated['grand_total'];
            } else {
                $grand_total = $purchase->grand_total;
            }
            if (isset($validated['paid'])) {
                $validated['due'] = isset($validated['due'])
                    ? $validated['due']
                    : round($grand_total - $validated['paid'], 2);
            }

            $purchase->update($validated);

            $purchaseArray = $purchase->toArray();
            $purchaseArray['products'] = json_decode($purchaseArray['products'], true);

            return response()->json([
                'status' => true,
                'message' => 'Purchase updated successfully',
                'data' => $purchaseArray
            ]);
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $ve->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified purchase from storage.
     */
    public function destroy($id)
    {
        try {
            $purchase = Purchase::find($id);

            if (!$purchase) {
                return response()->json([
                    'status' => false,
                    'message' => 'Purchase not found'
                ], 404);
            }

            $purchase->delete();

            return response()->json([
                'status' => true,
                'message' => 'Purchase deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Pay part or all of the due amount for a purchase.
     *
     * @param Request $request
     * @param int $id
     * @return \Illuminate\Http\JsonResponse
     */
    public function payDue(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'amount'   => 'required|numeric|min:0.01',
                'paid_by'  => 'required|string',
            ]);

            $purchase = Purchase::find($id);

            if (!$purchase) {
                return response()->json([
                    'status' => false,
                    'message' => 'Purchase not found',
                ], 404);
            }

            $due = floatval($purchase->due);
            $paid = floatval($purchase->paid);
            $amount = floatval($validated['amount']);

            if ($amount > $due) {
                return response()->json([
                    'status' => false,
                    'message' => 'Payment amount exceeds due.',
                ], 422);
            }

            $purchase->paid = round($paid + $amount, 2);
            $purchase->due  = round($purchase->grand_total - $purchase->paid, 2);
            $purchase->paid_by = $validated['paid_by'];

            $purchase->save();

            $purchaseArray = $purchase->toArray();
            $purchaseArray['products'] = json_decode($purchaseArray['products'], true);

            return response()->json([
                'status' => true,
                'message' => 'Due paid successfully',
                'data' => $purchaseArray
            ]);
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return response()->json([
                'status' => false,
                'message' => 'Validation failed',
                'errors' => $ve->errors(),
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
