<?php

namespace App\Http\Controllers;

use App\Models\Size;
use Illuminate\Http\Request;

class SizeController extends Controller
{
    /**
     * Display a listing of all sizes.
     */
    public function index(Request $request)
    {
        try {
            $query = Size::query();

            // 🔍 Search by name or code
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('code', 'like', "%{$search}%");
                });
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive'])) {
                $query->where('status', $request->status);
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $sizes = $query->paginate($perPage);

            // Transform size data
            $sizes->getCollection()->transform(function ($size) {
                return [
                    'id' => $size->id,
                    'name' => $size->name,
                    'code' => $size->code,
                    'description' => $size->description,
                    'status' => $size->status,
                    'created_at' => $size->created_at->toDateTimeString(),
                    'updated_at' => $size->updated_at->toDateTimeString(),
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Sizes retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $sizes->total(),
                    'total_page' => $sizes->lastPage(),
                    'current_page' => $sizes->currentPage(),
                    'current_page_count' => $sizes->count(),
                    'next_page' => $sizes->hasMorePages() ? $sizes->currentPage() + 1 : null,
                    'previous_page' => $sizes->onFirstPage() ? null : $sizes->currentPage() - 1,
                ],
                'data' => $sizes->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created size.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255|unique:sizes,name',
                'code' => 'nullable|string|max:10',
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $size = Size::create([
                'name' => $request->name,
                'code' => $request->code,
                'description' => $request->description,
                'status' => $request->status ?? 'active'
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Size created successfully',
                'data' => $size
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified size.
     */
    public function show($id)
    {
        try {
            $size = Size::find($id);

            if (!$size) {
                return response()->json(['status' => false, 'message' => 'Size not found'], 404);
            }

            return response()->json(['status' => true, 'data' => $size]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified size.
     */
    public function update(Request $request, $id)
    {
        try {
            $size = Size::find($id);

            if (!$size) {
                return response()->json(['status' => false, 'message' => 'Size not found'], 404);
            }

            $request->validate([
                'name' => 'sometimes|string|max:255|unique:sizes,name,' . $id,
                'code' => 'nullable|string|max:10',
                'description' => 'nullable|string',
                'status' => 'in:active,inactive'
            ]);

            $size->update([
                'name' => $request->name ?? $size->name,
                'code' => $request->code ?? $size->code,
                'description' => $request->description ?? $size->description,
                'status' => $request->status ?? $size->status
            ]);

            return response()->json([
                'status' => true,
                'message' => 'Size updated successfully',
                'data' => $size
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified size.
     */
    public function destroy($id)
    {
        try {
            $size = Size::find($id);

            if (!$size) {
                return response()->json(['status' => false, 'message' => 'Size not found'], 404);
            }

            $size->delete();

            return response()->json([
                'status' => true,
                'message' => 'Size deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
