<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Subcategory;
use Illuminate\Support\Facades\Storage;

class SubcategoryController extends Controller
{
    /**
     * Display a listing of subcategories.
     */
    public function index(Request $request)
    {
        try {
            $query = Subcategory::query();

            // 🔍 Search filter
            if ($request->filled('search')) {
                $search = $request->search;
                $query->where('name', 'like', "%{$search}%");
            }

            // 🧩 Status filter
            if ($request->filled('status') && in_array($request->status, ['active', 'inactive', 1, 0])) {
                $status = $request->status;
                if ($status === 'active' || $status === 1 || $status === '1') {
                    $query->where('status', 1);
                } else {
                    $query->where('status', 0);
                }
            }

            // ↕️ Sorting (default: desc)
            $order = $request->get('order', 'desc');
            $query->orderBy('id', $order);

            // 📄 Pagination (default: 10 per page)
            $perPage = $request->get('per_page', 10);
            $subcategories = $query->paginate($perPage);

            // 🖼️ Transform subcategory data
            $subcategories->getCollection()->transform(function ($subcategory) {
                return [
                    'id' => $subcategory->id,
                    'name' => $subcategory->name,
                    'category_id' => $subcategory->category_id,
                    'status' => $subcategory->status ? 'active' : 'inactive',
                    'description' => $subcategory->description,
                    'image' => $subcategory->image ? asset('storage/' . $subcategory->image) : null,
                    'category' => $subcategory->category ? [
                        'id' => $subcategory->category->id,
                        'name' => $subcategory->category->name,
                        'status' => $subcategory->category->status,
                        'image' => $subcategory->category->image ? asset('storage/' . $subcategory->category->image) : null,
                    ] : null,
                ];
            });

            return response()->json([
                'status' => true,
                'message' => 'Subcategories retrieved successfully',
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $subcategories->total(),
                    'total_page' => $subcategories->lastPage(),
                    'current_page' => $subcategories->currentPage(),
                    'current_page_count' => $subcategories->count(),
                    'next_page' => $subcategories->hasMorePages() ? $subcategories->currentPage() + 1 : null,
                    'previous_page' => $subcategories->onFirstPage() ? null : $subcategories->currentPage() - 1,
                ],
                'data' => $subcategories->items(),
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created subcategory.
     */
    public function store(Request $request)
    {
        try {
            $request->validate([
                'name' => 'required|string|max:255',
                'category_id' => 'required|exists:categories,id',
                'description' => 'nullable|string',
                'status' => 'nullable|in:active,inactive,1,0',
                'image' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
            ]);

            $imagePath = null;
            if ($request->hasFile('image')) {
                $imagePath = $request->file('image')->store('subcategories', 'public');
            }

            $status = ($request->status === null) ? 1
                    : (($request->status == 'active' || $request->status == 1 || $request->status === '1') ? 1 : 0);

            $subcategory = Subcategory::create([
                'name' => $request->name,
                'category_id' => $request->category_id,
                'description' => $request->description,
                'status' => $status,
                'image' => $imagePath,
            ]);

            $subcategoryData = [
                'id' => $subcategory->id,
                'name' => $subcategory->name,
                'category_id' => $subcategory->category_id,
                'description' => $subcategory->description,
                'status' => $subcategory->status ? 'active' : 'inactive',
                'image' => $subcategory->image ? asset('storage/' . $subcategory->image) : null,
            ];

            return response()->json([
                'status'=>true,
                'message' => 'Subcategory created successfully',
                'data' => $subcategoryData
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified subcategory.
     */
    public function show($id)
    {
        try {
            $subcategory = Subcategory::find($id);

            if (!$subcategory) {
                return response()->json(['status' => false, 'message' => 'Subcategory not found'], 404);
            }

            $subcategoryData = [
                'id' => $subcategory->id,
                'name' => $subcategory->name,
                'category_id' => $subcategory->category_id,
                'description' => $subcategory->description,
                'status' => $subcategory->status ? 'active' : 'inactive',
                'image' => $subcategory->image ? asset('storage/' . $subcategory->image) : null,
                'category' => $subcategory->category ? [
                    'id' => $subcategory->category->id,
                    'name' => $subcategory->category->name,
                    'status' => $subcategory->category->status,
                    'image' => $subcategory->category->image ? asset('storage/' . $subcategory->category->image) : null,
                ] : null,
            ];

            return response()->json(['status'=>true,'data' => $subcategoryData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified subcategory.
     */
    public function update(Request $request, $id)
    {
        try {
            $subcategory = Subcategory::find($id);

            if (!$subcategory) {
                return response()->json(['status' => false, 'message' => 'Subcategory not found'], 404);
            }

            $request->validate([
                'name' => 'nullable|string|max:255',
                'category_id' => 'nullable|exists:categories,id',
                'description' => 'nullable|string',
                'status' => 'nullable|in:active,inactive,1,0',
                'image' => 'nullable|image|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
            ]);

            if ($request->hasFile('image')) {
                // Delete old image
                if ($subcategory->image) {
                    Storage::disk('public')->delete($subcategory->image);
                }

                $subcategory->image = $request->file('image')->store('subcategories', 'public');
            }

            if ($request->name !== null) {
                $subcategory->name = $request->name;
            }
            if ($request->category_id !== null) {
                $subcategory->category_id = $request->category_id;
            }
            if ($request->description !== null) {
                $subcategory->description = $request->description;
            }
            if ($request->status !== null) {
                $subcategory->status = ($request->status == 'active' || $request->status == 1 || $request->status === '1') ? 1 : 0;
            }
            $subcategory->save();

            $subcategoryData = [
                'id' => $subcategory->id,
                'name' => $subcategory->name,
                'category_id' => $subcategory->category_id,
                'description' => $subcategory->description,
                'status' => $subcategory->status ? 'active' : 'inactive',
                'image' => $subcategory->image ? asset('storage/' . $subcategory->image) : null,
            ];

            return response()->json(['status'=>true,'message' => 'Subcategory updated successfully', 'data' => $subcategoryData]);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified subcategory.
     */
    public function destroy($id)
    {
        try {
            $subcategory = Subcategory::find($id);

            if (!$subcategory) {
                return response()->json(['status' => false, 'message' => 'Subcategory not found'], 404);
            }

            // Delete image
            if ($subcategory->image) {
                Storage::disk('public')->delete($subcategory->image);
            }

            $subcategory->delete();

            return response()->json(['status'=>true,'message' => 'Subcategory deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

}

