<?php

namespace App\Http\Controllers;

use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class SupplierController extends Controller
{
    /**
     * Display a listing of suppliers.
     */
    public function index(Request $request)
    {
        try {
            $query = Supplier::query();

            if ($request->filled('search')) {
                $search = $request->search;
                $query->where(function ($q) use ($search) {
                    $q->where('name', 'like', "%{$search}%")
                      ->orWhere('email', 'like', "%{$search}%")
                      ->orWhere('phone', 'like', "%{$search}%")
                      ->orWhere('company', 'like', "%{$search}%")
                      ->orWhere('address', 'like', "%{$search}%");
                });
            }

            $perPage = $request->get('per_page', 10);
            $suppliers = $query->orderBy('id', 'desc')->paginate($perPage);

            // Add full URL to image for each supplier
            $data = collect($suppliers->items())->map(function ($supplier) use ($request) {
                $arr = $supplier->toArray();
                if (!empty($arr['image'])) {
                    if (app()->runningInConsole() && !$request->server->get('HTTP_HOST')) {
                        // Keep it as a relative path if storage:link isn't set up for console
                    } else {
                        $arr['image'] = asset(Storage::url($supplier->image));
                    }
                }
                return $arr;
            });

            return response()->json([
                'status' => true,
                'message' => 'Suppliers retrieved successfully',
                'data' => $data,
                'pagination' => [
                    'limit_page' => $perPage,
                    'total_count' => $suppliers->total(),
                    'total_page' => $suppliers->lastPage(),
                    'current_page' => $suppliers->currentPage(),
                    'current_page_count' => $suppliers->count(),
                    'next_page' => $suppliers->hasMorePages() ? $suppliers->currentPage() + 1 : null,
                    'previous_page' => $suppliers->onFirstPage() ? null : $suppliers->currentPage() - 1,
                ]
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Store a newly created supplier in storage.
     */
    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name'      => 'required|string|max:255',
                'email'     => 'nullable|email|max:255',
                'phone'     => 'required|string|max:255|unique:suppliers,phone',
                'image'     => 'required|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'address'   => 'nullable|string|max:255',
                'company'   => 'nullable|string|max:255',
                'note'      => 'nullable|string',
            ]);

            if ($request->hasFile('image')) {
                $path = $request->file('image')->store('suppliers', 'public');
                $validated['image'] = $path;
            }

            $supplier = Supplier::create($validated);

            // Make image full URL
            $supplierArr = $supplier->toArray();
            if (!empty($supplierArr['image'])) {
                $supplierArr['image'] = asset(Storage::url($supplier->image));
            }

            return response()->json([
                'status' => true,
                'message' => 'Supplier created successfully',
                'data' => $supplierArr
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Display the specified supplier.
     */
    public function show($id)
    {
        try {
            $supplier = Supplier::find($id);

            if (!$supplier) {
                return response()->json([
                    'status' => false,
                    'message' => 'Supplier not found',
                ], 404);
            }

            // Make image full URL
            $supplierArr = $supplier->toArray();
            if (!empty($supplierArr['image'])) {
                $supplierArr['image'] = asset(Storage::url($supplier->image));
            }

            return response()->json([
                'status' => true,
                'message' => 'Supplier retrieved successfully',
                'data' => $supplierArr
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Update the specified supplier in storage.
     */
    public function update(Request $request, $id)
    {
        try {
            $supplier = Supplier::find($id);

            if (!$supplier) {
                return response()->json([
                    'status' => false,
                    'message' => 'Supplier not found'
                ], 404);
            }

            $validated = $request->validate([
                'name'      => 'sometimes|required|string|max:255',
                'email'     => 'nullable|email|max:255',
                'phone'     => [
                    'sometimes',
                    'required',
                    'string',
                    'max:255',
                ],
                'image'     => 'nullable|mimes:jpeg,jpg,png,bmp,gif,svg,webp,heic,heif,tiff,tif,ico,avif|max:20480',
                'address'   => 'nullable|string|max:255',
                'company'   => 'nullable|string|max:255',
                'note'      => 'nullable|string',
            ]);

            if ($request->hasFile('image')) {
                // Delete old image if exists
                if ($supplier->image && Storage::disk('public')->exists($supplier->image)) {
                    Storage::disk('public')->delete($supplier->image);
                }
                $path = $request->file('image')->store('suppliers', 'public');
                $validated['image'] = $path;
            }

            $supplier->update($validated);

            // Make image full URL
            $supplierArr = $supplier->toArray();
            if (!empty($supplierArr['image'])) {
                $supplierArr['image'] = asset(Storage::url($supplier->image));
            }

            return response()->json([
                'status' => true,
                'message' => 'Supplier updated successfully',
                'data' => $supplierArr
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }

    /**
     * Remove the specified supplier from storage.
     */
    public function destroy($id)
    {
        try {
            $supplier = Supplier::find($id);

            if (!$supplier) {
                return response()->json([
                    'status' => false,
                    'message' => 'Supplier not found'
                ], 404);
            }

            // Remove image if exists
            if ($supplier->image && Storage::disk('public')->exists($supplier->image)) {
                Storage::disk('public')->delete($supplier->image);
            }

            $supplier->delete();

            return response()->json([
                'status' => true,
                'message' => 'Supplier deleted successfully'
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'status' => false,
                'message' => $e->getMessage(),
                'errors' => [
                    'error' => [
                        $e->getMessage()
                    ]
                ]
            ], 500);
        }
    }
}
